define([
    'backbone',
    'moment',
    'Layout',
    'modules/new-appointment-request/resources/direct/preferred-date/time-slots-by-date-collection',
    'modules/new-appointment-request/views/direct/preferred-date/preferred-date-layout',
    'moment',
], function(Backbone, moment, Layout, TimeSlotsCollection, PreferredDateLayoutView) {
    'use strict';


    describe('Preferred Date Layout View', function() {
        var view;
        var $view;
        var model;
        var today = moment();
        var todayString;
        var timeSlotsByDateCollection;
        var layout;

        beforeEach(function() {
            todayString = today.format('MM/DD/YYYY');
            timeSlotsByDateCollection = new TimeSlotsCollection([
                {startDateTime: todayString + ' 14:30:00'},
                {startDateTime: todayString + ' 15:00:00'},
                {startDateTime: todayString + ' 15:30:00'},
            ]);

            layout = new Layout();
            layout.render();

            spyOn(PreferredDateLayoutView.prototype, 'showSelectedDateTimeView');
            model = new Backbone.Model();
            view = new PreferredDateLayoutView({
                model: model,
                collection: timeSlotsByDateCollection,
            });

            layout.showChildView('content', view);
            $view = view.$el;
        });

        afterEach(function() {
            layout.getRegion('content').empty();
        });

        describe('on show', function() {
            it('has regions to show the form sub-sections', function() {
                expect($view.has(view.regions.preferredDateRegion).length).toEqual(1);
                expect($view.has(view.regions.availableDatesTimesRegion).length).toEqual(1);
                expect($view.has(view.regions.selectedDateTimeSlotRegion).length).toEqual(1);
                expect($view.has(view.regions.ariaLiveRegion).length).toEqual(1);
            });

            it('has a model with desired date undefined', function() {
                expect(view.model.has('desiredDate')).toBe(false);
            });

            it('has a hidden Show All Dates button', function () {
                var $btn = $view.find('#show-all-dates-btn:hidden');
                expect($btn.length).toEqual(1);
                expect($btn.text()).toEqual('Show All Dates');
            });
        });

        describe('when the desired date is undefined', function() {
            it('has a disabled Show Availability button', function () {
                var $btn = $view.find('#show-availability-btn');
                expect($btn.length).toEqual(1);
                expect($btn.text()).toEqual('Show Availability');
                expect($btn.prop('disabled')).toBe(true);
                expect($btn.attr('aria-disabled')).toBe('true');
                expect($btn.hasClass('ui-state-disabled')).toBe(true);
                expect($btn.attr('aria-label')).toEqual('Enter a preferred date to enable Show Availability button');
            });
        });

        describe('when the desired date is given', function() {
            beforeEach(function() {
                model.set('desiredDate', todayString);
            });

            it('has an enabled Show Availability button', function () {
                var $btn = $view.find('#show-availability-btn');
                expect($btn.length).toEqual(1);
                expect($btn.text()).toEqual('Show Availability');
                expect($btn.prop('disabled')).toBe(false);
                expect($btn.attr('aria-disabled')).toBe('false');
                expect($btn.hasClass('ui-state-disabled')).toBe(false);
                expect($btn.attr('aria-label')).toBe(undefined);
            });
        });

        describe('when the desired date is available and the Show Availability button is clicked', function() {
            beforeEach(function() {
                spyOn(PreferredDateLayoutView.prototype, 'showPreferredDateAvailableView');
                model.set('desiredDate', todayString);

                $view.find('#show-availability-btn').trigger('click');
            });

            it('calls showPreferredDateAvailableView()', function () {
                expect(view.showPreferredDateAvailableView).toHaveBeenCalled();
            });

            it('shows the Show All Dates button', function () {
                var $btn = $view.find('#show-all-dates-btn:visible');
                expect($btn.length).toEqual(1);
                expect($btn.text()).toEqual('Show All Dates');
            });
            // Show All Dates functionality tested in automated tests
        });

        describe('when the desired date is not available and the Show Availability button is clicked', function() {
            beforeEach(function() {
                spyOn(PreferredDateLayoutView.prototype, 'showClosestDatesView');
                model.set('desiredDate', today.add(2, 'days').format('MM/DD/YYYY'));

                $view.find('#show-availability-btn').trigger('click');
            });

            it('calls showClosestDatesView()', function () {
                expect(view.showClosestDatesView).toHaveBeenCalled();
            });

            it('shows the Show All Dates button', function () {
                var $btn = $view.find('#show-all-dates-btn:visible');
                expect($btn.length).toEqual(1);
                expect($btn.text()).toEqual('Show All Dates');
            });
            // Show All Dates functionality tested in automated tests
        });

        // showing Selected Date Time View tested in automated tests

        describe('when the model has been updated with a selected time slot', function() {
            beforeEach(function() {
                model.set('dateTime', '12/02/2017 08:30:00');
            });

            it('updates the aria message', function () {
                expect(view.getRegion('ariaLiveRegion').$el.text()).toEqual('You are scheduling an appointment for: Saturday, 12/02/2017 08:30');
            });
        });
    });
});
